package de.yanenko.igor;

import java.awt.BasicStroke;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.geom.Point2D;
import java.util.Iterator;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSpinner;
import javax.swing.JTextField;
import javax.swing.JToolTip;
import javax.swing.SpinnerListModel;
import javax.swing.SpinnerModel;
import javax.swing.SpinnerNumberModel;
import javax.swing.border.TitledBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import prefuse.Constants;
import prefuse.Display;
import prefuse.Visualization;
import prefuse.action.Action;
import prefuse.action.ActionList;
import prefuse.action.ItemAction;
import prefuse.action.RepaintAction;
import prefuse.action.animate.ColorAnimator;
import prefuse.action.animate.LocationAnimator;
import prefuse.action.animate.QualityControlAnimator;
import prefuse.action.animate.VisibilityAnimator;
import prefuse.action.assignment.ColorAction;
import prefuse.action.assignment.DataColorAction;
import prefuse.action.assignment.FontAction;
import prefuse.action.assignment.SizeAction;
import prefuse.action.assignment.StrokeAction;
import prefuse.action.filter.FisheyeTreeFilter;
import prefuse.action.layout.CollapsedSubtreeLayout;
import prefuse.action.layout.graph.NodeLinkTreeLayout;
import prefuse.activity.SlowInSlowOutPacer;
import prefuse.controls.ControlAdapter;
import prefuse.controls.FocusControl;
import prefuse.controls.PanControl;
import prefuse.controls.ToolTipControl;
import prefuse.controls.WheelZoomControl;
import prefuse.data.Tree;
import prefuse.data.Tuple;
import prefuse.data.io.TreeMLReader;
import prefuse.data.tuple.TupleSet;
import prefuse.render.DefaultRendererFactory;
import prefuse.render.EdgeRenderer;
import prefuse.render.AbstractShapeRenderer;
import prefuse.render.LabelRenderer;
import prefuse.util.ColorLib;
import prefuse.util.FontLib;
import prefuse.util.ui.JSearchPanel;
import prefuse.util.ui.UILib;
import prefuse.visual.EdgeItem;
import prefuse.visual.NodeItem;
import prefuse.visual.VisualItem;
import prefuse.visual.expression.InGroupPredicate;
import prefuse.visual.sort.TreeDepthItemSorter;

/**
 * <p>Search tree visualization based on the Prefuse visualization kit.</p>
 * 
 * <p>For more information
 * and demos on Prefuse visit <a href="http://www.prefuse.org">www.prefuse.org</a>
 * </p>
 * 
 * @version 1.0
 * @author <a href="http://www.yanenko.org">Olga Yanenko</a>
 */
public class HypoTree extends Display {
    
	/** The constant serialVersionUID. */
	private static final long serialVersionUID = 1L;
	
	/** The constant tree. It's needed for addressing the whole tree structure.  */
	private static final String tree = "tree";
    
    /** The constant treeNodes. It's needed for addressing the tree nodes. */
    private static final String treeNodes = "tree.nodes";
    
    /** The constant treeEdges. It's needed for addressing the tree edges. */
    private static final String treeEdges = "tree.edges";
    
    /** The maximum amount of available node weights (size of the weight vector). */
    public static int MAX_WEIGHT;
    
    /** The node renderer. */
    private static LabelRenderer m_nodeRenderer;
    
    /** The edge renderer. */
    private static EdgeRenderer m_edgeRenderer;
    
    /** The attribute used for node labeling. */
    private String m_label;
    
    /** The orientation of the tree. */
    private static int m_orientation = Constants.ORIENT_LEFT_RIGHT;
    
    /** The constant gray which is the color for node borders and edges. */
    private static final Color gray = ColorLib.getColor(120, 120, 120);
    
    /** The constant stroke which is the standard stroke for node borders and edges. */
    private static final BasicStroke stroke = new BasicStroke(2);
    
    /** The constant fatStroke which is used for focused nodes. */
    private static final BasicStroke fatStroke = new BasicStroke(8);
    
    /** The constant middleStroke which is used for result nodes. */
    private static final BasicStroke middleStroke = new BasicStroke(4);
    
    /** The constant font which is the standard font for the visualization. */
    private static final Font font = FontLib.getFont("Tahoma", 10);
    
    /** The part of the weight vector which is currently displayed. */
    private static int weight = 4;
    
    /** The constant fill which is the standard fill action for the nodes. */
    private static final NodeDataColorAction fill = new NodeDataColorAction();
    
    /** The constant nodeToolTip which is the action for showing tool tips. */
    private static final NodeToolTipControl nodeToolTip = new NodeToolTipControl("weight" + weight);
    
    /**
     * Instantiates a new hypotheses tree.
     * 
     * @param t the tree
     * @param label the attribute
     */
    public HypoTree(Tree t, String label) {
        super(new Visualization());
        m_label = label;
        // Add the tree to the visualization.
        m_vis.add(tree, t);
        // Create a new node renderer.
        m_nodeRenderer = new LabelRenderer(m_label);
        m_nodeRenderer.setRenderType(AbstractShapeRenderer.RENDER_TYPE_DRAW_AND_FILL);
        m_nodeRenderer.setHorizontalAlignment(Constants.CENTER);
        m_nodeRenderer.setHorizontalPadding(10);
        m_nodeRenderer.setVerticalPadding(6);
        m_nodeRenderer.setRoundedCorner(20, 20);
        // Create a new edge renderer.
        m_edgeRenderer = new EdgeRenderer(Constants.EDGE_TYPE_CURVE);
        // Create the renderer factory.
        DefaultRendererFactory rf = new DefaultRendererFactory(m_nodeRenderer);
        rf.add(new InGroupPredicate(treeEdges), m_edgeRenderer);
        m_vis.setRendererFactory(rf);
        // Check the weight vector size an set the maximum. 
        Iterator i = m_vis.getGroup(treeNodes).tuples();
        Tuple item = (Tuple) i.next();
        int a = 1;
        while (item.canGetString("weight" + a)) {
        	a++;
        }
        MAX_WEIGHT = a - 1;   
        // Create different actions for the tree visualization.
        ItemAction textColor = new TextColorAction(treeNodes);
        ItemAction edgeStrokeColor = new EdgeColorAction(treeEdges);
        ItemAction nodeStroke = new NodeStrokeAction(treeNodes);
        ItemAction edgeStroke = new EdgeStrokeAction(treeEdges);
        ItemAction fontAction = new FontAction(treeNodes, font);
        ItemAction sizeAction = new NodeSizeAction(treeNodes, 1.0);
        ItemAction nodeStrokeColor = new NodeStrokeColorAction(treeNodes);
        // Create the tree layout.
        NodeLinkTreeLayout treeLayout = new NodeLinkTreeLayout(tree, m_orientation, 60, 10, 20);
        treeLayout.setLayoutAnchor(new Point2D.Double(200, 150));
        m_vis.putAction("treeLayout", treeLayout);
        // Create the subtree layout.
        CollapsedSubtreeLayout subLayout = new CollapsedSubtreeLayout(tree, m_orientation);
        m_vis.putAction("subLayout", subLayout);
        // Create the auto pan. 
        AutoPanAction autoPan = new AutoPanAction();
        // Create the filtering and layout.
        ActionList filter = new ActionList();
        filter.add(new FisheyeTreeFilter(tree, Visualization.SELECTED_ITEMS, 1));
        filter.add(autoPan);
        filter.add(fontAction);
        filter.add(treeLayout);
        filter.add(subLayout);
        filter.add(textColor);
        filter.add(nodeStroke);
        filter.add(edgeStroke);
        filter.add(sizeAction);
        filter.add(edgeStrokeColor);
        filter.add(nodeStrokeColor);
        filter.add(fill);
        m_vis.putAction("filter", filter);
        // Create the animated transition.
        ActionList animate = new ActionList(1000);
        animate.setPacingFunction(new SlowInSlowOutPacer());
        animate.add(autoPan);
        animate.add(new VisibilityAnimator(tree));
        animate.add(new LocationAnimator(treeNodes));
        animate.add(new ColorAnimator(treeNodes));
        animate.add(new RepaintAction());
        animate.add(new QualityControlAnimator());
        m_vis.putAction("animate", animate);
        m_vis.alwaysRunAfter("filter", "animate");
        // Initialize the display.
        setSize(700,500);
        setItemSorter(new TreeDepthItemSorter());
        addControlListener(new WheelZoomControl());
        addControlListener(new PanControl());
        addControlListener(nodeToolTip);
        addControlListener(new EdgeToolTipControl("operator"));
        addControlListener(new NodeFocusControl(1, "filter"));
        // Filter the tree and perform layout.
        setOrientation(m_orientation, m_vis);
        m_vis.run("filter");
    }
    
    /**
     * Sets the orientation.
     * 
     * @param orientation the new orientation
     */
    public static void setOrientation(int orientation, Visualization vis) {
        NodeLinkTreeLayout rtl = (NodeLinkTreeLayout)vis.getAction("treeLayout");
        CollapsedSubtreeLayout stl = (CollapsedSubtreeLayout)vis.getAction("subLayout");
        switch (orientation) {
        case Constants.ORIENT_LEFT_RIGHT:
            m_nodeRenderer.setHorizontalAlignment(Constants.LEFT);
            m_edgeRenderer.setHorizontalAlignment1(Constants.RIGHT);
            m_edgeRenderer.setHorizontalAlignment2(Constants.LEFT);
            m_edgeRenderer.setVerticalAlignment1(Constants.CENTER);
            m_edgeRenderer.setVerticalAlignment2(Constants.CENTER);
            break;
        case Constants.ORIENT_RIGHT_LEFT:
            m_nodeRenderer.setHorizontalAlignment(Constants.RIGHT);
            m_edgeRenderer.setHorizontalAlignment1(Constants.LEFT);
            m_edgeRenderer.setHorizontalAlignment2(Constants.RIGHT);
            m_edgeRenderer.setVerticalAlignment1(Constants.CENTER);
            m_edgeRenderer.setVerticalAlignment2(Constants.CENTER);
            break;
        case Constants.ORIENT_TOP_BOTTOM:
            m_nodeRenderer.setHorizontalAlignment(Constants.CENTER);
            m_edgeRenderer.setHorizontalAlignment1(Constants.CENTER);
            m_edgeRenderer.setHorizontalAlignment2(Constants.CENTER);
            m_edgeRenderer.setVerticalAlignment1(Constants.BOTTOM);
            m_edgeRenderer.setVerticalAlignment2(Constants.TOP);
            break;
        case Constants.ORIENT_BOTTOM_TOP:
            m_nodeRenderer.setHorizontalAlignment(Constants.CENTER);
            m_edgeRenderer.setHorizontalAlignment1(Constants.CENTER);
            m_edgeRenderer.setHorizontalAlignment2(Constants.CENTER);
            m_edgeRenderer.setVerticalAlignment1(Constants.TOP);
            m_edgeRenderer.setVerticalAlignment2(Constants.BOTTOM);
            break;
        default:
            throw new IllegalArgumentException(
                "Unrecognized orientation value: " + orientation);
        }
        m_orientation = orientation;
        rtl.setOrientation(orientation);
        stl.setOrientation(orientation);
    }
    
    /**
     * Gets the orientation.
     * 
     * @return the orientation
     */
    public int getOrientation() {
        return m_orientation;
    }
    
    /**
     * Hides the children of a node item.
     * 
     * @param i the node item
     * @param ts the tuple set
     */
    public static void hideChildren(NodeItem i, TupleSet ts) {
    	ts.removeTuple(i);
    	Iterator children = i.children();
    	// Recursively hide all children.
    	while (children.hasNext()) {
    		hideChildren((NodeItem)children.next(), ts);
    	}
    }
    
    /**
     * Shows all children of a node item.
     * 
     * @param i the node item
     * @param ts the tuple set
     */
    public static void showChildren(NodeItem i, TupleSet ts) {
    	ts.addTuple(i);
    	Iterator children = i.children();
    	// Recursively hide all children.
    	while (children.hasNext()) {
    		showChildren((NodeItem)children.next(), ts);
    	}
    }
    
    /**
     * Shows parents of a node item.
     * 
     * @param i the node item
     * @param ts the tuple set
     */
    public static void showParent(NodeItem i, TupleSet ts) {
    	ts.addTuple(i);
    	NodeItem item = (NodeItem) i.getParent();
    	if (item != null) {
    		showParent(item, ts);
    	}
    }
    
    /**
     * Shows the tree view.
     * 
     * @param datafile the file containing the TreeML structure
     * 
     * @return the JComponent containing the tree view
     */
    public static JComponent show(String datafile) {
        Color BACKGROUND = Color.WHITE;
        Color FOREGROUND = Color.BLACK;
        // Create tree from file.
        Tree t = null;
        try {
            t = (Tree)new TreeMLReader().readGraph(datafile);
        } catch ( Exception e ) {
            e.printStackTrace();
            System.exit(1);
        }
        // Create a new tree map from the tree.
        final HypoTree tview = new HypoTree(t, "name");
        tview.setBackground(BACKGROUND);
        tview.setForeground(FOREGROUND);
        // Create a swing label containing for displaying the node information.
        final JLabel title = new JLabel("                 ");
        title.setFont(font);
        title.setBackground(BACKGROUND);
        title.setForeground(FOREGROUND);
        title.setOpaque(true);
        title.setSize(700, 100);
        title.setVerticalAlignment(JLabel.TOP);
        // Create a spinner panel for choosing the weight and orientation to display.
        JPanel spinnerPane = new JPanel(new BorderLayout());
        spinnerPane.setBackground(BACKGROUND);
        spinnerPane.setOpaque(true);
        spinnerPane.setSize(700, 50);
        // Create the weight spinner model.
        SpinnerModel model = new SpinnerNumberModel(weight, 1, MAX_WEIGHT, 1);
        // Create the weight spinner.
        final JSpinner spinner = new JSpinner(model);
        spinner.setFont(font);
        spinner.setBackground(BACKGROUND);
        spinner.setForeground(FOREGROUND);
        spinner.setBorder(BorderFactory.createMatteBorder(2, 2, 2, 2, gray));
        // Add a change listener, which changes the nodes fill color and tool tips according to the chosen weight.
        spinner.addChangeListener(new ChangeListener() {
			public void stateChanged(ChangeEvent e) {
				Visualization vis = tview.getVisualization();
				int i = (Integer) spinner.getModel().getValue();
				weight = i;
				fill.setDataField("weight" + i);
				nodeToolTip.setLabel("weight" + i);
				vis.run("filter");
			}
        });
        // Add the spinner to the spinner panel.
        spinnerPane.add(spinner, BorderLayout.LINE_END);
        // Create the orientation spinner model.
        SpinnerModel model_o = new SpinnerListModel(new Object[] {"horizontal", "vertical"});
        // Create the orientation spinner.
        final JSpinner spinner_o = new JSpinner(model_o);
        spinner_o.setFont(font);
        spinner_o.setBackground(BACKGROUND);
        spinner_o.setForeground(FOREGROUND);
        spinner_o.setBorder(BorderFactory.createMatteBorder(2, 2, 2, 2, gray));
        // Add a change listener, which changes the orientation.
        spinner_o.addChangeListener(new ChangeListener() {
			public void stateChanged(ChangeEvent e) {
				Visualization vis = tview.getVisualization();
				String s = spinner_o.getModel().getValue().toString();
				if (s.equals("horizontal")) {
					m_orientation = Constants.ORIENT_LEFT_RIGHT;
				} else if (s.equals("vertical")) {
					m_orientation = Constants.ORIENT_TOP_BOTTOM;
				}
				setOrientation(m_orientation, vis);
				vis.run("filter");
			}
        });
        // Add the spinner to the spinner panel.
        spinnerPane.add(spinner_o, BorderLayout.LINE_START);
        // Create a search panel.
        JPanel search = new JPanel(new BorderLayout());
        search.setBackground(BACKGROUND);
        JLabel label = new JLabel("search: ");
        label.setBackground(BACKGROUND);
        final JLabel result = new JLabel("");
        result.setBackground(BACKGROUND);
        JTextField textfield = new JTextField(5);
        textfield.setBorder(BorderFactory.createMatteBorder(2, 2, 2, 2, gray));
        textfield.setFont(font);
        textfield.addKeyListener(new KeyAdapter() {
			public void keyTyped(KeyEvent e) {
				int key = e.getKeyChar();
			    if (key == KeyEvent.VK_ENTER) {
			    	JTextField tf = (JTextField)e.getSource();
			    	String text = tf.getText();
			    	System.out.println("Searching for node no. " + text + " ...");
			    	result.setText("Searching...");
			    	if (text.matches("[0-9]*")) {
			    		Visualization vis = tview.getVisualization();
			    		Iterator i = vis.getGroup(treeNodes).tuples();
			    		boolean found = false;
			    		while (i.hasNext() && !found) {
			    			NodeItem item = (NodeItem)i.next();
			    			if (item.canGetString("name")) {
			    				if (item.getString("name").equals(text)) {
			    					found = true;
			    					System.out.println("Found!");
			    					result.setText("Found!");
			    					vis.getFocusGroup(Visualization.FOCUS_ITEMS).clear();
			    					vis.getFocusGroup(Visualization.FOCUS_ITEMS).addTuple(item);
			    					showParent(item, vis.getGroup(Visualization.SELECTED_ITEMS));
						    		vis.run("filter");
			    				}
			    			}
			    		}
			    		if (!found) {
			    			result.setText("Not found!");
			    			System.out.println("Not found!");
			    		}
			    	}
			    }
			}
        });
        label.setFont(font);
        result.setFont(font);
        JPanel flow = new JPanel();
        flow.setBackground(BACKGROUND);
        flow.add(label);
        flow.add(textfield);
        flow.add(result);
        search.add(flow, BorderLayout.CENTER);
        // Add the search panel to the spinner panel.
        spinnerPane.add(search, BorderLayout.CENTER);  
        // Add a mouse click listener to the tree view.
        tview.addControlListener(new ControlAdapter() {
            public void itemClicked(VisualItem item, MouseEvent e) {
            	Visualization vis = item.getVisualization();
                TupleSet ts = vis.getGroup(Visualization.SELECTED_ITEMS);
                // Hide the children of a node, if a right click is done on this node.
            	if (UILib.isButtonPressed(e, RIGHT_MOUSE_BUTTON)) {
            		hideChildren((NodeItem)item, ts);
            		vis.run("filter");
            	} else {
            		// Show the hypothesis of a node in the node info label.
            		if (item.canGetString("text")) {
                    	title.setText("<html>" + item.getString("text").replaceAll("\t", "<br/>") + "</html>");
                    	title.setVisible(true);
                    }
            		if (e.getClickCount() == 2) {
            			showChildren((NodeItem)item, ts);
            			vis.run("filter");
            		} else {
            			if (!item.isInGroup(Visualization.SELECTED_ITEMS)) {
                			ts.addTuple(item);
                		}
            		}
            	}
            }
        });
        // Create a scroll pane for the node info label.
        JScrollPane scrPane = new JScrollPane(title,JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
        scrPane.setBackground(BACKGROUND);
        scrPane.setPreferredSize(new Dimension(700,100));
        scrPane.setBorder(BorderFactory.createTitledBorder(BorderFactory.createMatteBorder(2, 2, 2, 2, gray), "Node Info", TitledBorder.DEFAULT_JUSTIFICATION,TitledBorder.DEFAULT_POSITION, font));
        // Create the main panel. The tree view is placed in the center,
        // the spinner on the top right and the node info at the bottom.
        JPanel panel = new JPanel(new BorderLayout());
        panel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
        panel.setBackground(BACKGROUND);
        panel.setForeground(FOREGROUND);
        panel.add(tview, BorderLayout.CENTER);
        panel.add(scrPane, BorderLayout.SOUTH);
        panel.add(spinnerPane, BorderLayout.NORTH);
        return panel;
    }
    /**
     * The class AutoPanAction which smoothly moves the tree, so that the selected
     * item is placed in the center of the screen.
     */
    public class AutoPanAction extends Action {
        
        /** The start point. */
        private Point2D m_start = new Point2D.Double();
        
        /** The end point. */
        private Point2D m_end   = new Point2D.Double();
        
        /** The current position. */
        private Point2D m_cur   = new Point2D.Double();
        
        /** The bias. */
        private int     m_bias  = 0;
        
        /* (non-Javadoc)
         * @see prefuse.action.Action#run(double)
         */
        public void run(double frac) {
            TupleSet ts = m_vis.getFocusGroup(Visualization.FOCUS_ITEMS);
            if (ts.getTupleCount() == 0)
                return;
            if (frac == 0.0) {
                int xbias = 0, ybias = 0;
                switch (m_orientation) {
                case Constants.ORIENT_LEFT_RIGHT:
                    xbias = m_bias;
                    break;
                case Constants.ORIENT_RIGHT_LEFT:
                    xbias = -m_bias;
                    break;
                case Constants.ORIENT_TOP_BOTTOM:
                    ybias = m_bias;
                    break;
                case Constants.ORIENT_BOTTOM_TOP:
                    ybias = -m_bias;
                    break;
                }

                VisualItem vi = (VisualItem)ts.tuples().next();
                m_cur.setLocation(getWidth()/2, getHeight()/2);
                getAbsoluteCoordinate(m_cur, m_start);
                m_end.setLocation(vi.getX() + xbias, vi.getY() + ybias);
            } else {
                m_cur.setLocation(m_start.getX() + frac*(m_end.getX()-m_start.getX()),
                                  m_start.getY() + frac*(m_end.getY()-m_start.getY()));
                panToAbs(m_cur);
            }
        }
    }
    
    /**
     * The class NodeToolTipControl which controls the tool tips for the nodes.
     * It shows the content of the node weight attribute, which is currently displayed.
     */
    public static class NodeToolTipControl extends ToolTipControl {
    	
	    /** The attribute for displaying in the tool tip. */
	    private String label;
        
        /**
         * Instantiates a new node tool tip control.
         * 
         * @param field the attribute to show in the tool tip
         */
        public NodeToolTipControl(String field) {
            super(field);
            label = field;
        }
        
        /**
         * Sets the label.
         * 
         * @param field the new label
         */
        public void setLabel(String field) {
        	label = field;
        }
    	
    	/* (non-Javadoc)
	     * @see prefuse.controls.ToolTipControl#itemEntered(prefuse.visual.VisualItem, java.awt.event.MouseEvent)
	     */
	    public void itemEntered(VisualItem item, MouseEvent e) {
    		Display d = (Display)e.getSource();
    		// Create a custom tool tip an set the colors, border and font.
    		JToolTip tool = new JToolTip();
        	tool.setBackground(Color.WHITE);
        	tool.setBorder(BorderFactory.createLineBorder(gray, 2));
        	tool.setFont(font);
        	tool.setVisible(true);
        	d.setCustomToolTip(tool);
            if (item.canGetString(label)) {
                d.setToolTipText(item.getString(label));
            }
        }
    }
    
    /**
     * The class EdgeToolTipControl which controls the tool tips for the edges.
     * It shows the operator which produced the child hypothesis.
     */
    public static class EdgeToolTipControl extends ToolTipControl {
    	
	    /** The attribute to show. */
	    private String label;
        
        /**
         * Instantiates a new edge tool tip control.
         * 
         * @param field the attribute to display in the tool tip.
         */
        public EdgeToolTipControl(String field) {
            super(field);
            label = field;
        }
    	
    	/* (non-Javadoc)
	     * @see prefuse.controls.ToolTipControl#itemEntered(prefuse.visual.VisualItem, java.awt.event.MouseEvent)
	     */
	    public void itemEntered(VisualItem item, MouseEvent e) {
    		if (item.isInGroup(treeEdges)) {
	    		NodeItem son = ((EdgeItem) item).getTargetItem();
	    		Display d = (Display)e.getSource();
	    		// Create a custom tool tip an set the colors, border and font.
	    		JToolTip tool = new JToolTip();
	        	tool.setBackground(Color.WHITE);
	        	tool.setBorder(BorderFactory.createLineBorder(gray, 2));
	        	tool.setFont(font);
	        	tool.setVisible(true);
	        	d.setCustomToolTip(tool);
	            if (son.canGetString(label)) {
	                d.setToolTipText(son.getString(label));
	            }
    		}
        }
    }
    
    /**
     * The class NodeSizeAction. It makes the leaf nodes smaller than the branches.
     */
    public static class NodeSizeAction extends SizeAction {
    	
    	/**
	     * Instantiates a new node size action.
	     * 
	     * @param group the node group
	     * @param size the default size
	     */
	    public NodeSizeAction(String group, double size) {
    		super(group, size);
    	}
    	
    	/* (non-Javadoc)
	     * @see prefuse.action.assignment.SizeAction#getSize(prefuse.visual.VisualItem)
	     */
	    public double getSize(VisualItem item) {
            if (((NodeItem) item).getChildCount() == 0) {
                return new Double(0.6);
            } else {
            	return this.getDefaultSize();
            } 
        }
    }
    
    /**
     * The class NodeStrokeAction. It shows a fat border when a node is focused.
     */
    public static class NodeStrokeAction extends StrokeAction {
    	
    	/**
	     * Instantiates a new node stroke action.
	     * 
	     * @param group the node group
	     */
	    public NodeStrokeAction(String group) {
            super(group);
        }
    	
    	/* (non-Javadoc)
	     * @see prefuse.action.assignment.StrokeAction#getStroke(prefuse.visual.VisualItem)
	     */
	    public BasicStroke getStroke(VisualItem item) {
    		if (m_vis.isInGroup(item, Visualization.FOCUS_ITEMS)) {
    			return fatStroke;
    		} else {
    			if (item.canGetString("text")) {
    				if (!item.getString("text").contains("*")) {
    					return middleStroke;
    				} else {
    					return stroke;
    				}
    			} else {
    				return stroke;
    			}
    		}
        }
    }
    
    /**
     * The class EdgeStrokeAction. It shows different strokes for the different operators.
     */
    public static class EdgeStrokeAction extends StrokeAction {
    	
	    /** The stroke for operator FOLD. */
	    private BasicStroke stroke1;
    	
	    /** The stroke for operator MTCH. */
	    private BasicStroke stroke2;
    	
	    /** The stroke for operator PART. */
	    private BasicStroke stroke3;
    	
	    /** The stroke for operator CALL. */
	    private BasicStroke stroke4;
    	
	    /** The stroke for operator NAIVE_MAP. */
	    private BasicStroke stroke5;
    	
    	/**
	     * Instantiates a new edge stroke action.
	     * 
	     * @param group the group
	     */
	    public EdgeStrokeAction(String group) {
            super(group);
            float[] dash1 = new float[2];
			dash1[0] = 5; //sichtbar
			dash1[1] = 5; //unsichtbar
			float[] dash2 = new float[2];
			dash2[0] = 1; //sichtbar
			dash2[1] = 5; //unsichtbar
			float[] dash3 = new float[2];
			dash3[0] = 10; //sichtbar
			dash3[1] = 5; //unsichtbar
			float[] dash4 = new float[4];
			dash4[0] = 5; //sichtbar
			dash4[1] = 5; //unsichtbar
			dash4[2] = 1; //sichtbar
			dash4[3] = 5; //unsichtbar

            stroke1 = new BasicStroke(2, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND, 1f, dash1, 0);
            stroke2 = new BasicStroke(2, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND, 1f, dash4, 0);
            stroke3 = new BasicStroke(2, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND, 1f, dash2, 0);
            stroke4 = new BasicStroke(2, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND, 1f, dash3, 0);
            stroke5 = new BasicStroke(2, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND, 1f, null, 0);
        }
    	
    	/* (non-Javadoc)
	     * @see prefuse.action.assignment.StrokeAction#getStroke(prefuse.visual.VisualItem)
	     */
	    public BasicStroke getStroke(VisualItem item) {
    		NodeItem son = ((EdgeItem) item).getTargetItem();
    		// Set the default stroke for operators which were not considered.
    		BasicStroke result = stroke;
    		if (son.canGetString("operator")) {
    			String op = son.getString("operator");
    			if (op != null) {
	    			if (op.equals("FOLD")) {
	    				return stroke1;
	    			} else if (son.getString("operator").equals("MTCH")) {
	    				return stroke2;
	    			} else if (son.getString("operator").equals("PART")) {
	    				return stroke3;
	    			} else if (son.getString("operator").equals("CALL")) {
	    				return stroke4;
	    			} else if (son.getString("operator").equals("NAIVE_MAP")) {
	    				return stroke5;
	    			}
    			}
    		}
    		return result;
        }
    }
    
    /**
     * The class TextColorAction. It sets the text color of branch nodes to white
     * and of leaf nodes to transparent.
     */
    public static class TextColorAction extends ColorAction {
    	
    	/**
	     * Instantiates a new text color action.
	     * 
	     * @param group the node group
	     */
	    public TextColorAction(String group) {
            super(group, VisualItem.TEXTCOLOR);
        }
        
        /* (non-Javadoc)
         * @see prefuse.action.assignment.ColorAction#getColor(prefuse.visual.VisualItem)
         */
        public int getColor(VisualItem item) {
            String name = item.getString("name");
            if (name.equals("targets") || name.equals("H")) {
            	return ColorLib.alpha(0);
            } else {
            	return ColorLib.rgb(0, 0, 0);
            }
        }
    }
    
    /**
     * The class NodeStrokeColorAction. It sets the stroke color to gray
     * except the 'targets' node which should not be displayed.
     */
    public static class NodeStrokeColorAction extends ColorAction {
    	
    	/**
	     * Instantiates a new node stroke color action.
	     * 
	     * @param group the node group
	     */
	    public NodeStrokeColorAction(String group) {
            super(group, VisualItem.STROKECOLOR);
        }
        
        /* (non-Javadoc)
         * @see prefuse.action.assignment.ColorAction#getColor(prefuse.visual.VisualItem)
         */
        public int getColor(VisualItem item) {
        	int color = ColorLib.color(gray);
            if (item.canGetString("name")) {
            	if (item.getString("name").equals("targets")) {
            		color = ColorLib.rgb(255, 255, 255);
            	}
            }
            return color;
        }
    }
    
    /**
     * The class EdgeColorAction. It shows different operator edges in different gray colors. 
     */
    public static class EdgeColorAction extends ColorAction {
    	
    	/**
	     * Instantiates a new edge color action.
	     * 
	     * @param group the node group
	     */
	    public EdgeColorAction(String group) {
            super(group, VisualItem.STROKECOLOR);
        }
        
        /* (non-Javadoc)
         * @see prefuse.action.assignment.ColorAction#getColor(prefuse.visual.VisualItem)
         */
        public int getColor(VisualItem item) {
        	NodeItem son = ((EdgeItem) item).getTargetItem();
        	int color = ColorLib.color(gray);
            if (son.canGetString("operator")) {
            	if (son.getString("operator").equals("CALL")) {
            		color = ColorLib.rgb(150, 150, 150);
            	} else if (son.getString("operator").equals("MTCH")) {
            		color = ColorLib.rgb(125, 125, 125);
            	} else if (son.getString("operator").equals("PART")) {
            		color = ColorLib.rgb(100, 100, 100);
            	} else if (son.getString("operator").equals("FOLD")) {
            		color = ColorLib.rgb(75, 75, 75);
            	} else if (son.getString("operator").equals("NAIVE_MAP")) {
            		color = ColorLib.rgb(50, 50, 50);
            	} else if (son.getString("operator").equals("")) {
            		color = ColorLib.rgb(255, 255, 255);
            	}
            }
            return color;
        }
    }
    
    /**
     * The Class NodeDataColorAction. It fills the nodes with colors of
     * the green to red palette according to the weight.
     */
    public static class NodeDataColorAction extends DataColorAction {
    	
		/**
		 * Instantiates a new node data color action.
		 */
		public NodeDataColorAction() {
			super(treeNodes, "weight" + weight, Constants.ORDINAL, VisualItem.FILLCOLOR);
		}
    	
		/* (non-Javadoc)
		 * @see prefuse.action.assignment.DataColorAction#createPalette(int)
		 */
		protected int[] createPalette(int size) {
			int[] palette = new int[size];
			float step = (float) (0.30 / (size-2));
			float h = (float) 0.30;
			palette[0] = ColorLib.rgb(255, 255, 255);
			for (int i=1; i<size; i++) {
				palette[i] = ColorLib.hsb(h, 1.0f, 1.0f);
				h = h - step;
				if (h < 0) {
					h = 0;
				}
			}
			return palette;
	    }
    }
    
    /**
     * The class NodeFocusControl. The focus control ignores the tree edges and reacts
     * only to clicks on tree nodes.
     */
    public static class NodeFocusControl extends FocusControl{
    	
    	/**
	     * Instantiates a new node focus control.
	     * 
	     * @param clicks the click count
	     * @param act the action to perform
	     */
	    public NodeFocusControl(int clicks, String act) {
    		super(clicks, act);
    	}
    	
    	/* (non-Javadoc)
	     * @see prefuse.controls.FocusControl#filterCheck(prefuse.visual.VisualItem)
	     */
	    protected boolean filterCheck(VisualItem item) {
            if (item.isInGroup(treeNodes)) {
            	return true;
            } else {
            	return false;
            }
	    }
    }
}
