package de.yanenko.igor;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.jdom.*;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

/**
 * <p>Class Parser, which extracts the information about the hypotheses created by IGOR
 * for later visualization. This information is brought in the TreeML format,
 * because this format is required by the Prefuse visualization kit. A file with
 * the same name as the input file and file extension .tml is created in the input
 * directory.
 * The manipulation of XML documents and elements is performed by JDOM.
 * </p>
 * <p>For more information about JDOM see <a href="http://www.jdom.org">www.jdom.org</a>
 * </p>
 * <p>For more information
 * and demos on Prefuse visit <a href="http://www.prefuse.org">www.prefuse.org</a>
 * </p>
 * <p>For more information about the IGOR-Software please visit <a href="http://www.cogsys.wiai.uni-bamberg.de/effalip/">www.cogsys.wiai.uni-bamberg.de/effalip/</a>.
 * </p>
 * 
 * @version 1.0
 * @author <a href="http://www.yanenko.de">Olga Yanenko</a>
 */
public class Parser {
	
	/** The log file produced by IGOR. */
	private File file;
	
	/** The target functions the log file contains. */
	private String[] targets;
	
	/** The steps in each target. */
	private ArrayList<String>[] steps;
	
	/** The BufferedReader for the file. */
	private BufferedReader reader;
	
	/** The name for the output file in TreeML format. */
	private String output;
	
	/** The XML Document containing the hypotheses in a hierarchical structure. */
	private Document doc;
	
	/** The variable to check whether the BufferedReader reached the end of the logfile part dealing with one target. */
	private boolean stop = false;
	
	/**
	 * Instantiates a new parser.
	 * 
	 * @param input the String for the input file
	 * @param out the String for the output file
	 */
	public Parser(String input, String out) {
		super();
		file = new File(input);
		output = out;
	}
	
	/**
	 * Runs the Parser. First the log file is read and the hypotheses stored in a XML document.
	 * Then the Document is brought in TreeML format and saved to a file.
	 */
	protected void run() {
		try {
			reader = new BufferedReader(new FileReader(file));
			read();
		} catch (FileNotFoundException e) {
			System.out.println("Log-file doesn't exist!!");
		}
		makeLeafs(doc.getRootElement());
		writeXML(xmlToTreeml(), output);
	}
	
	/**
	 * Reads the log file and extracts the hypotheses.
	 */
	private void read() {
		doc = new Document();
		// Find the targets names.
		findTargets();
		steps = new ArrayList[targets.length];
		// Create some elements for attribute declarations in TreeML.
		Element root = new Element("tree");
		this.doc.addContent(root);
		Element dec = new Element("declarations");
		root.addContent(dec);
		Element attdec1 = new Element("attributeDecl");
		attdec1.setAttribute("name", "name");
		attdec1.setAttribute("type", "String");
		Element attdec2 = new Element("attributeDecl");
		attdec2.setAttribute("name", "text");
		attdec2.setAttribute("type", "String");
		Element attdec3 = new Element("attributeDecl");
		attdec3.setAttribute("name", "operator");
		attdec3.setAttribute("type", "String");
		dec.addContent(attdec1);
		dec.addContent(attdec2);
		dec.addContent(attdec3);
		// Create first (root) element named 'targets'.
		Element first = new Element("branch");
		first.setAttribute("name", "targets");
		first.setAttribute("operator", "");
		root.addContent(first);
		// Loop for parsing each target.
		for (int i=0; i<targets.length; i++) {
			stop = false;
			// Create element for target no. i, which is the parent for all according hypotheses.
			Element target = new Element("branch");
			target.setAttribute("name", targets[i]);
			target.setAttribute("operator", "");
			first.addContent(target);
			// Extract the hypotheses for target no. i.
			steps[i] = new ArrayList<String>(0);
			findSteps(i, target);
		}
	}
	
	/**
	 * Extract the hypotheses of a target.
	 * 
	 * @param i the number of target
	 * @param el the element which contains the target information
	 */
	private void extractContent(int i, Element el) {
		Iterator<String> iter = steps[i].iterator();
		int count = 1;
		while (iter.hasNext()) {
			String content = iter.next();
			if (!content.equals("")) {
				String[] split1 = content.split("of Hypo [(]");
				Element dad;
				String a = split1[1];
				// Split the String in one part containing the parent hypothesis and a second part
				// containing the child hypotheses.
				String[] split2 = a.split("resulted in [0-9]* new hypotheses[.]");
				// Extract the parent hypothesis.
				String[] hypo = extractHypo(split2[0]);
				// Check if the parent hypothesis already exists.
				dad = findElement(el, hypo[1]);
				// Insert parent hypothesis if it's not already in the document.
				if (dad == null) {
					dad = insertElement(hypo[1], hypo[0], "",  el);
					// Make TreeML attribute declarations for the values in the weight vector.
					if (count == 1) {
						String[] weights = hypo[0].split(",");
						Element decl = doc.getRootElement().getChild("declarations");
						for (int j=0; j<weights.length; j++) {
							Element weight = new Element("attributeDecl");
							weight.setAttribute("name", "weight" + (j + 1));
							weight.setAttribute("type", "String");
							decl.addContent(weight);
						}
					}
				}
				// Set the counter as name for the parent hypothesis.
				dad.setAttribute("name", "" + count);
				String op = split2[1].substring(6, split2[1].length() - 50);
				// Split the child hypothesis part in several parts belonging to the different operators.
				String[] operators1 = op.split("OP[0-9]+[a-z]?: ");
				for (String b : operators1) {
					if (!b.equals("")) {
						// Split the string in the operator name and the according hypotheses.
						String[] operators2 = b.split("Advancements:.*Hypotheses[ \t]+:");
						// String containing the hypotheses.
						String hypos = operators2[1];
						hypos = hypos.substring(4, hypos.length() - 4);
						if (!hypos.equals("")) {
							// Name of the operator.
							String operator = operators2[0];
							hypos = hypos.substring(6);
							// Split the different hypotheses.
							String[] operators3 = hypos.split(",Hypo [(]");
							// Insert the elements.
							for (String c : operators3) {
								hypo = extractHypo(c);
								insertElement(hypo[1], hypo[0], operator, dad);
							}
						}
					}
				}
			}
			count++;
		}
	}
	
	/**
	 * Extracts the hypothesis and its weight from a String of the form
	 * 'x,x,x,x,(x))(*lengths,{a0 = a1})(*fun1,{a0 = a1})...'.
	 * 
	 * @param s the input String
	 * 
	 * @return the String array containing the weight and the hypothesis
	 */
	private String[] extractHypo(String s) {
		String[] result = new String[2];
		// Split the string in a weight and a hypothesis part.
		String[] split1 = s.split("[)][)][(]");
		String[] split2 = split1[0].split("[,][(]");
		result[0] = split2[0];
		result[1] = split1[1].substring(0, split1[1].length() - 1).replaceAll("[)][(]", "\t").replaceAll("", "").trim();
		return result;
	}
	
	/**
	 * Inserts an element.
	 * 
	 * @param b the hypothesis
	 * @param w the weight
	 * @param op the operator
	 * @param parent the parent element
	 * 
	 * @return the element
	 */
	private Element insertElement(String b, String w, String op, Element parent) {
		Element element = findElement(doc.getRootElement(), b);
		// Insert the element if it doesn't already exist.
		if (element == null) {
			element = new Element("branch");
			element.setText(b);
			element.setAttribute("operator", op);
			String[] weights = w.split(",");
			for (int i=0; i < weights.length; i++) {
				element.setAttribute("weight" + (i + 1), weights[i]);
			}
			element.setAttribute("name", "H");
			parent.addContent(element);
		}
		return element;
	}
	
	/**
	 * Finds the element which contains the given hypothesis.
	 * 
	 * @param el the parent element
	 * @param text the hypothesis
	 * 
	 * @return the element
	 */
	private Element findElement(Element el, String text) {
		Element r = null;
		// List all children of the parent element.
		List<Element> l = el.getChildren();
		// Check if one of the child elements contains the hypothesis.
		for (int i=0; i<l.size(); i++) {
			if (r == null) {
				if (l.get(i).getTextTrim().equals(text.trim())) {
					r = l.get(i);
				} else {
					// Recursively search child elements.
					r = findElement(l.get(i), text);
				}
			}
		}
		return r;
	}
	
	/**
	 * Finds the targets of the log file.
	 */
	private void findTargets() {
		String s = findLine("Targets.*");
		s = s.replaceFirst("Targets", "").trim();
		s = s.substring(1, s.length()-1);
		targets = s.split("' '");
	}
	
	/**
	 * Finds the steps of target no. i.
	 */
	private void findSteps(int i, Element el) {
		String s = findLine(".*STARTING SYNTHESIS for TARGET '" + targets[i] + "'.*");
		int step = 1;
		while (s != null && !stop) {
			findLine("- - Entering Rule-Advancement-Loop for the " + step + ". time - -");
			try {
				s = reader.readLine();
				s = reader.readLine();
			} catch (IOException e) {
				System.out.println("Cannot read line!");
			}
			step++;
			boolean record = false;
			String r = "";
			while (s != null && !stop && !s.trim().matches("^-( -){23}")) {
				if (!stop && !s.trim().matches(".*Results  Results  Results  Results.*")) {
					try {
						s = reader.readLine();
						if (s.matches("^- - Summary.*")) {
							record = true;
						}
						if (record == true) {
							r = r + "" + s.trim();
						}
					} catch (IOException e) {
						System.out.println("Cannot read line!");
					}
				} else {
					stop = true;
				}
			}
			steps[i].add(r);
		}
		extractContent(i, el);
	}
	
	/**
	 * Finds the line containing a regular expression.
	 * 
	 * @param regex the regular expression
	 * 
	 * @return the line containing the regular expression wanted
	 */
	private String findLine(String regex) {
		String s = "";
		while (s != null && !s.trim().matches(regex) && !stop) {
			try {
				s = reader.readLine();
				if (s.matches(".*Stopped after entering loop.*") || s.matches(".*Results  Results  Results  Results.*") || s.matches(".*SEARCHSPACE EXHAUSTED.*")) {
					stop = true;
				}
			} catch (IOException e) {
				System.out.println("Failure reading line!");
			}
		}
		return s;
	}
	
	/**
	 * Writes the XML file.
	 * 
	 * @param d the document
	 * @param file the file name
	 */
	private void writeXML(Document d, String file) {
		// Set the format (line breaks and tabs)
		Format format = Format.getPrettyFormat();
		XMLOutputter xo = new XMLOutputter(format);
		try {
			xo.output(d, new FileOutputStream(file));
		} catch (FileNotFoundException e) {
			System.out.println("File or folder doesn't exist!");
		} catch (IOException e) {
			System.out.println("Failure writing file!");
		}
	}
	
	/**
	 * Changes an element without children to a TreeML leaf.
	 * 
	 * @param el the element
	 */
	private void makeLeafs(Element el) {
		List<Element> l = el.getChildren();
		if (l.isEmpty() && el.getName().equals("branch")) {
			el.setName("leaf");
		} else {
			for (int i=0; i<l.size(); i++) {
				makeLeafs(l.get(i));
			}
		}
	}
	
	/**
	 * Brings the XML document in a TreeML structure.
	 * 
	 * @return the document
	 */
	private Document xmlToTreeml() {
		Document temp = doc;
		Element root = temp.getRootElement();
		changeElement(root);
		return temp;
	}
	
	/**
	 * Brings an XML element in a TreeML structure.
	 * 
	 * @param el the element
	 */
	private void changeElement(Element el) {
		if (el.getName().equals("branch") || el.getName().equals("leaf")) {
			List<Attribute> a = el.getAttributes();
			int size = a.size();
			// Replaces the attributes by attribute child elements.
			for (int i=size-1; i>=0; i--) {
				Attribute aa = a.get(i);
				Element att = new Element("attribute");
				att.setAttribute("name", aa.getName());
				att.setAttribute("value", aa.getValue());
				el.addContent(att);
				el.removeAttribute(aa);
			}
			// Creates an attribute child element containing the hypothesis.
			Element text = new Element("attribute");
			text.setAttribute("name", "text");
			text.setAttribute("value", el.getText());
			el.addContent(text);
		}
		// If the element has children recursively change the children's structure.
		List<Element> l = el.getChildren();
		if(!l.isEmpty()) {
			for (int i=0; i<l.size(); i++) {
				changeElement(l.get(i));
			}
		}
	}
}
